


<?php 
ob_start();
include("../config/db.php");
include("../config/session_user.php");

// Check session
if(!isset($_SESSION['user_id'])){
    header("Location: login.php");
    exit;
}
$user_id = (int)$_SESSION['user_id'];

/* ================= ONE SOURCE OF TRUTH ================= */
function getRunningGameId($conn){
    $g = mysqli_fetch_assoc(mysqli_query($conn,"
        SELECT id FROM games 
        WHERE status='running' 
        ORDER BY id DESC 
        LIMIT 1
    "));
    if(!$g){
        mysqli_query($conn,"INSERT INTO games(status,created_at) VALUES('running',NOW())");
        return mysqli_insert_id($conn);
    }
    return (int)$g['id'];
}

/* ================= PLACE BET (🔥 SMART DEDUCTION SYSTEM) ================= */
if(isset($_POST['action']) && $_POST['action']=="place_bet"){
    header("Content-Type: application/json");
    ob_clean();

    $game_id = getRunningGameId($conn);
    $bets = $_POST['bets'] ?? [];

    if(empty($bets)){
        echo json_encode(["status"=>"error","msg"=>"No bets provided"]);
        exit;
    }

    // Add recharge_amount column if not exists
    mysqli_query($conn, "ALTER TABLE users ADD COLUMN IF NOT EXISTS recharge_amount DECIMAL(10,2) DEFAULT 0.00");
    mysqli_query($conn, "ALTER TABLE users ADD COLUMN IF NOT EXISTS winning_amount DECIMAL(10,2) DEFAULT 0.00");

    // Get user data with separate amounts
    $u = mysqli_fetch_assoc(mysqli_query($conn,"
        SELECT wallet, 
               COALESCE(recharge_amount, 0) as recharge_amount, 
               COALESCE(winning_amount, 0) as winning_amount 
        FROM users WHERE id=$user_id
    "));
    
    if(!$u){
        echo json_encode(["status"=>"error","msg"=>"User not found"]);
        exit;
    }

    // If amounts are not set, calculate them from wallet and bet history
    if($u['recharge_amount'] == 0 && $u['winning_amount'] == 0 && $u['wallet'] > 0) {
        // Calculate winning amount from bet history
        $total_winnings = mysqli_fetch_assoc(mysqli_query($conn, "
            SELECT COALESCE(SUM(win_amount), 0) as total_winnings 
            FROM bets 
            WHERE user_id = $user_id AND status = 'win'
        "));
        
        $winning_amt = $total_winnings['total_winnings'];
        $recharge_amt = $u['wallet'] - $winning_amt;
        
        // Update the separate amounts
        mysqli_query($conn, "
            UPDATE users 
            SET recharge_amount = $recharge_amt, winning_amount = $winning_amt 
            WHERE id = $user_id
        ");
        
        $u['recharge_amount'] = $recharge_amt;
        $u['winning_amount'] = $winning_amt;
    }

    $totalAmount = 0;
    $validBets = [];
    
    // Parse bets
    foreach($bets as $betJson){
        $bet = json_decode($betJson, true);
        if($bet && isset($bet['color']) && isset($bet['amount'])){
            $color = trim($bet['color']);
            $amount = (int)$bet['amount'];
            
            // Skip empty colors and invalid amounts
            if(!empty($color) && $amount >= 10){
                $validBets[] = [
                    'color' => strtolower($color),
                    'amount' => $amount
                ];
                $totalAmount += $amount;
            }
        }
    }

    if(empty($validBets)){
        echo json_encode(["status"=>"error","msg"=>"No valid bets"]);
        exit;
    }

    if($u['wallet'] < $totalAmount){
        echo json_encode(["status"=>"error","msg"=>"Insufficient balance. Need ₹$totalAmount"]);
        exit;
    }

    // SMART DEDUCTION: First from recharge, then from winning
    $remaining_amount = $totalAmount;
    $deduct_from_recharge = 0;
    $deduct_from_winning = 0;
    
    if($u['recharge_amount'] >= $remaining_amount) {
        // Can deduct everything from recharge amount
        $deduct_from_recharge = $remaining_amount;
        $deduct_from_winning = 0;
    } else {
        // Deduct all recharge amount, rest from winning
        $deduct_from_recharge = $u['recharge_amount'];
        $deduct_from_winning = $remaining_amount - $u['recharge_amount'];
    }

    // Update user amounts with smart deduction
    mysqli_query($conn,"
        UPDATE users 
        SET wallet = wallet - $totalAmount,
            recharge_amount = recharge_amount - $deduct_from_recharge,
            winning_amount = winning_amount - $deduct_from_winning
        WHERE id = $user_id
    ");

    // Insert bets
    $insertedBets = [];
    foreach($validBets as $bet){
        $color = mysqli_real_escape_string($conn, $bet['color']);
        $amount = $bet['amount'];
        
        $result = mysqli_query($conn,"
            INSERT INTO bets(user_id,game_id,color,amount,status,win_amount,created_at)
            VALUES($user_id,$game_id,'$color',$amount,'pending',0,NOW())
        ");
        
        if($result){
            $insertedBets[] = [
                'id' => mysqli_insert_id($conn),
                'color' => $color,
                'amount' => $amount
            ];
        }
    }

    echo json_encode([
        "status" => "success",
        "wallet" => $u['wallet'] - $totalAmount,
        "msg" => "Bets placed successfully!",
        "bets_placed" => count($insertedBets),
        "total_amount" => $totalAmount,
        "game_id" => $game_id,
        "inserted_bets" => $insertedBets,
        "deduction_info" => [
            "total_bet" => $totalAmount,
            "from_recharge" => $deduct_from_recharge,
            "from_winning" => $deduct_from_winning,
            "remaining_recharge" => $u['recharge_amount'] - $deduct_from_recharge,
            "remaining_winning" => $u['winning_amount'] - $deduct_from_winning,
            "smart_deduction_used" => ($deduct_from_recharge > 0 && $deduct_from_winning > 0)
        ]
    ]);
    exit;
}

/* ================= TEST WINNING (FOR DEBUGGING) ================= */
if(isset($_POST['action']) && $_POST['action']=="test_win"){
    header("Content-Type: application/json");
    ob_clean();
    
    $game_id = (int)($_POST['game_id'] ?? 0);
    $force_winner = $_POST['force_winner'] ?? null;
    
    if(!$game_id){
        echo json_encode(["status"=>"error","msg"=>"Invalid game ID"]);
        exit;
    }
    
    // Force a specific winner for testing
    $colors=['red','green','yellow','blue','purple','orange','pink','cyan'];
    $winner = $force_winner && in_array($force_winner, $colors) ? $force_winner : $colors[array_rand($colors)];
    
    // Get user's bets
    $bets = mysqli_query($conn,"SELECT * FROM bets WHERE game_id=$game_id AND user_id=$user_id AND status='pending'");
    $winAmount = 0;
    $totalBet = 0;
    $betDetails = [];
    
    while($b = mysqli_fetch_assoc($bets)){
        $totalBet += $b['amount'];
        $betDetails[] = [
            'color' => $b['color'],
            'amount' => $b['amount'],
            'winner' => $winner,
            'match' => (strtolower($b['color']) === strtolower($winner))
        ];
        
        if(strtolower($b['color']) === strtolower($winner)){
            $winAmount += ($b['amount'] * 7);
        }
    }
    
    echo json_encode([
        "winner" => $winner,
        "win_amount" => $winAmount,
        "total_bet" => $totalBet,
        "bet_details" => $betDetails,
        "test_mode" => true
    ]);
    exit;
}

/* ================= FORCE WIN TEST (FOR DEBUGGING) ================= */
if(isset($_POST['action']) && $_POST['action']=="force_win"){
    header("Content-Type: application/json");
    ob_clean();
    
    $game_id = (int)($_POST['game_id'] ?? 0);
    $force_color = $_POST['force_color'] ?? 'red';
    
    // Set winner to the forced color
    $winner = $force_color;
    
    // Get user's bets
    $bets_result = mysqli_query($conn, "SELECT * FROM bets WHERE game_id=$game_id AND user_id=$user_id AND color != '' ORDER BY id ASC");
    
    $winAmount = 0;
    $winningBets = [];
    
    if($bets_result){
        while($bet = mysqli_fetch_assoc($bets_result)){
            if(strtolower(trim($bet['color'])) === strtolower(trim($winner))){
                $individualWin = $bet['amount'] * 7;
                $winAmount += $individualWin;
                
                mysqli_query($conn,"UPDATE users SET wallet = wallet + $individualWin, winning_amount = winning_amount + $individualWin WHERE id = $user_id");
                mysqli_query($conn,"UPDATE bets SET status='win', win_amount=$individualWin WHERE id={$bet['id']}");
                
                $winningBets[] = [
                    'color' => $bet['color'],
                    'bet_amount' => $bet['amount'],
                    'win_amount' => $individualWin
                ];
            }
        }
    }
    
    $user = mysqli_fetch_assoc(mysqli_query($conn,"SELECT wallet FROM users WHERE id=$user_id"));
    
    echo json_encode([
        "winner" => $winner,
        "result" => "win",
        "win_amount" => $winAmount,
        "wallet" => (int)$user['wallet'],
        "winning_bets" => $winningBets,
        "forced_test" => true
    ]);
    exit;
}

/* ================= BUSINESS ALGORITHM - SMART WINNER SELECTION ================= */
function calculateSmartWinner($conn, $game_id) {
    // Get all pending bets for this game from ALL users
    $all_bets = mysqli_query($conn, "
        SELECT color, SUM(amount) as total_amount, COUNT(*) as bet_count
        FROM bets 
        WHERE game_id = $game_id AND status = 'pending'
        GROUP BY color
        ORDER BY total_amount DESC
    ");
    
    $color_stats = [];
    $total_wagered = 0;
    $colors = ['red','green','yellow','blue','purple','orange','pink','cyan'];
    
    // Initialize all colors with 0
    foreach($colors as $color) {
        $color_stats[$color] = ['amount' => 0, 'count' => 0];
    }
    
    // Fill actual bet data
    if($all_bets && mysqli_num_rows($all_bets) > 0) {
        while($bet = mysqli_fetch_assoc($all_bets)) {
            $color_stats[$bet['color']] = [
                'amount' => (float)$bet['total_amount'],
                'count' => (int)$bet['bet_count']
            ];
            $total_wagered += (float)$bet['total_amount'];
        }
    }
    
    // If no bets, return random color
    if($total_wagered == 0) {
        return $colors[array_rand($colors)];
    }
    
    // BUSINESS ALGORITHM: Calculate potential losses for each color
    $potential_losses = [];
    $payout_multiplier = 7; // 7x payout instead of 2x
    
    foreach($colors as $color) {
        $bet_amount = $color_stats[$color]['amount'];
        $potential_payout = $bet_amount * $payout_multiplier;
        $house_loss = $potential_payout - $total_wagered;
        
        $potential_losses[$color] = [
            'bet_amount' => $bet_amount,
            'potential_payout' => $potential_payout,
            'house_loss' => $house_loss,
            'profit_margin' => (($total_wagered - $potential_payout) / $total_wagered) * 100
        ];
    }
    
    // Sort colors by house profit (ascending house loss = more profit)
    uasort($potential_losses, function($a, $b) {
        return $a['house_loss'] <=> $b['house_loss'];
    });
    
    // SMART SELECTION ALGORITHM
    $profitable_colors = [];
    $neutral_colors = [];
    $loss_colors = [];
    
    foreach($potential_losses as $color => $stats) {
        if($stats['house_loss'] <= 0) {
            // House profits or breaks even
            $profitable_colors[] = $color;
        } elseif($stats['house_loss'] <= ($total_wagered * 0.1)) {
            // Small loss (within 10% of total wagered)
            $neutral_colors[] = $color;
        } else {
            // Big loss for house
            $loss_colors[] = $color;
        }
    }
    
    // SELECTION PROBABILITY (House always has advantage)
    $rand = mt_rand(1, 100);
    
    if($rand <= 70 && !empty($profitable_colors)) {
        // 70% chance: Select most profitable color
        return $profitable_colors[0];
    } elseif($rand <= 85 && !empty($profitable_colors)) {
        // 15% chance: Select random profitable color
        return $profitable_colors[array_rand($profitable_colors)];
    } elseif($rand <= 95 && !empty($neutral_colors)) {
        // 10% chance: Select neutral color (small loss)
        return $neutral_colors[array_rand($neutral_colors)];
    } else {
        // 5% chance: Let users win big (for excitement and retention)
        if(!empty($loss_colors)) {
            return $loss_colors[array_rand($loss_colors)];
        } else {
            return $colors[array_rand($colors)];
        }
    }
}

/* ================= UNPREDICTABLE PATTERN SYSTEM ================= */
function addRandomnessPattern($base_winner, $game_id) {
    // Add some randomness to make it unpredictable
    $colors = ['red','green','yellow','blue','purple','orange','pink','cyan'];
    
    // Get last 5 results to avoid obvious patterns
    $recent_results = mysqli_query($GLOBALS['conn'], "
        SELECT winning_color 
        FROM games 
        WHERE status='completed' AND winning_color IS NOT NULL
        ORDER BY completed_at DESC 
        LIMIT 5
    ");
    
    $recent_colors = [];
    if($recent_results) {
        while($row = mysqli_fetch_assoc($recent_results)) {
            $recent_colors[] = $row['winning_color'];
        }
    }
    
    // If same color appeared 3+ times in last 5 games, reduce its probability
    $color_count = array_count_values($recent_colors);
    if(isset($color_count[$base_winner]) && $color_count[$base_winner] >= 3) {
        // 30% chance to override with different color
        if(mt_rand(1, 100) <= 30) {
            $alternative_colors = array_diff($colors, [$base_winner]);
            return $alternative_colors[array_rand($alternative_colors)];
        }
    }
    
    return $base_winner;
}

/* ================= RESULT (🔥 BUSINESS ALGORITHM) ================= */
if(isset($_POST['action']) && $_POST['action']=="result"){
    header("Content-Type: application/json");
    ob_clean();

    try {
        $game_id = (int)($_POST['game_id'] ?? 0);
        if(!$game_id){
            echo json_encode(["status"=>"error","msg"=>"Invalid game ID"]);
            exit;
        }

        // 🎯 SMART BUSINESS ALGORITHM - Calculate optimal winner
        $smart_winner = calculateSmartWinner($conn, $game_id);
        $winner = addRandomnessPattern($smart_winner, $game_id);
        
        // Check database connection
        if(!$conn){
            echo json_encode(["status"=>"error","msg"=>"Database connection failed"]);
            exit;
        }
        
        // Debug: Check what bets exist
        $all_bets_check = mysqli_query($conn, "SELECT * FROM bets WHERE game_id=$game_id AND user_id=$user_id");
        if(!$all_bets_check){
            echo json_encode(["status"=>"error","msg"=>"Database query failed: " . mysqli_error($conn)]);
            exit;
        }
        
        $all_bets_count = mysqli_num_rows($all_bets_check);
        
        // Get pending bets only
        $bets_result = mysqli_query($conn, "SELECT * FROM bets WHERE game_id=$game_id AND user_id=$user_id AND status='pending'");
        if(!$bets_result){
            echo json_encode(["status"=>"error","msg"=>"Pending bets query failed: " . mysqli_error($conn)]);
            exit;
        }
        
        $pending_bets_count = mysqli_num_rows($bets_result);

        $winAmount = 0;
        $totalBetAmount = 0;
        $winningBets = [];
        $processedBets = [];

        // Process each pending bet
        if($pending_bets_count > 0){
            while($bet = mysqli_fetch_assoc($bets_result)){
                $totalBetAmount += $bet['amount'];
                
                $betColor = strtolower(trim($bet['color']));
                $winnerColor = strtolower(trim($winner));
                
                $processedBets[] = [
                    'bet_id' => $bet['id'],
                    'bet_color' => $betColor,
                    'winner_color' => $winnerColor,
                    'amount' => $bet['amount'],
                    'match' => ($betColor === $winnerColor)
                ];
                
                if($betColor === $winnerColor){
                    // WIN - 7x payout (Business Algorithm)
                    $win = $bet['amount'] * 7;
                    $winAmount += $win;
                    
                    // Add to wallet and winning amount
                    $wallet_update = mysqli_query($conn, "UPDATE users SET wallet = wallet + $win, winning_amount = winning_amount + $win WHERE id = $user_id");
                    if(!$wallet_update){
                        echo json_encode(["status"=>"error","msg"=>"Wallet update failed: " . mysqli_error($conn)]);
                        exit;
                    }
                    
                    // Update bet to win
                    $bet_update = mysqli_query($conn, "UPDATE bets SET status='win', win_amount=$win WHERE id={$bet['id']}");
                    if(!$bet_update){
                        echo json_encode(["status"=>"error","msg"=>"Bet update failed: " . mysqli_error($conn)]);
                        exit;
                    }
                    
                    $winningBets[] = [
                        'color' => $bet['color'],
                        'bet_amount' => $bet['amount'],
                        'win_amount' => $win
                    ];
                } else {
                    // LOSS - Update bet to loss
                    mysqli_query($conn, "UPDATE bets SET status='loss', win_amount=0 WHERE id={$bet['id']}");
                }
            }
        }

        // Mark current game as completed WITH winning_color
        $winner_escaped = mysqli_real_escape_string($conn, $winner);
        $game_complete = mysqli_query($conn, "UPDATE games SET status='completed', winning_color='$winner_escaped', completed_at=NOW() WHERE id=$game_id");
        if(!$game_complete){
            echo json_encode(["status"=>"error","msg"=>"Game completion failed: " . mysqli_error($conn)]);
            exit;
        }
        
        // Create fresh new game
        $new_game = mysqli_query($conn, "INSERT INTO games(status,created_at) VALUES('running',NOW())");
        if(!$new_game){
            echo json_encode(["status"=>"error","msg"=>"New game creation failed: " . mysqli_error($conn)]);
            exit;
        }
        $new_game_id = mysqli_insert_id($conn);

        // Get updated wallet
        $user_result = mysqli_query($conn, "SELECT wallet FROM users WHERE id=$user_id");
        if(!$user_result){
            echo json_encode(["status"=>"error","msg"=>"User wallet query failed: " . mysqli_error($conn)]);
            exit;
        }
        $user = mysqli_fetch_assoc($user_result);

        // Simple, clean response (no extra data)
        $response = [
            "winner" => $winner,
            "result" => $winAmount > 0 ? "win" : "loss",
            "win_amount" => $winAmount,
            "total_bet" => $totalBetAmount,
            "wallet" => (int)$user['wallet'],
            "new_game_id" => $new_game_id
        ];
        
        // Clear any output buffer
        while(ob_get_level()) {
            ob_end_clean();
        }
        
        echo json_encode($response);
        exit;
        
    } catch(Exception $e) {
        while(ob_get_level()) {
            ob_end_clean();
        }
        echo json_encode(["status"=>"error","msg"=>"Exception: " . $e->getMessage()]);
        exit;
    }
}

/* ================= PAGE LOAD ================= */
$game_id = getRunningGameId($conn);
$user = mysqli_fetch_assoc(mysqli_query($conn,"SELECT wallet FROM users WHERE id=$user_id"));
$colors=['red','green','yellow','blue','purple','orange','pink','cyan'];
?>




<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
<title>Color Betting Game</title>
<style>
body{background:linear-gradient(135deg,#667eea 0%,#764ba2 100%);color:#fff;font-family:'Segoe UI',Arial;margin:0;min-height:100vh}
.header{padding:15px 20px;display:flex;justify-content:space-between;align-items:center;background:rgba(0,0,0,0.3);backdrop-filter:blur(10px);position:relative}
.header-left{display:flex;align-items:center;flex:1}
.header-center{display:flex;align-items:center;justify-content:center;flex:1}
.header-right{display:flex;align-items:center;justify-content:flex-end;flex:1}
.balance{color:#ffd700;font-size:18px;font-weight:bold;text-shadow:0 2px 4px rgba(0,0,0,0.5)}
.user-info{color:#fff;opacity:0.9;font-size:14px}
.logout-btn{background:rgba(220,53,69,0.9);color:#fff;border:none;padding:8px 16px;border-radius:8px;cursor:pointer;text-decoration:none;font-size:13px;transition:all 0.3s ease}
.logout-btn:hover{background:rgba(220,53,69,1);transform:translateY(-1px)}

.timer-section{text-align:center;padding:20px;background:rgba(255,255,255,0.1);margin:10px;border-radius:15px;backdrop-filter:blur(5px)}
.timer{font-size:48px;font-weight:bold;color:#ffd700;text-shadow:0 0 20px rgba(255,215,0,0.5);margin:10px 0}
.timer.warning{color:#ff4757;animation:pulse 1s infinite}

@keyframes pulse{0%,100%{transform:scale(1)}50%{transform:scale(1.1)}}

.colors{display:grid;grid-template-columns:repeat(4,1fr);gap:15px;padding:20px;max-width:600px;margin:0 auto}
.color{padding:25px;border-radius:15px;font-weight:bold;text-align:center;cursor:pointer;opacity:0.8;position:relative;transition:all 0.3s ease;box-shadow:0 4px 15px rgba(0,0,0,0.2);color:#fff;text-shadow:0 1px 3px rgba(0,0,0,0.5)}
.color:hover{opacity:1;transform:translateY(-3px);box-shadow:0 8px 25px rgba(0,0,0,0.3)}
.color.active{outline:4px solid #ffd700;opacity:1;transform:scale(1.05);box-shadow:0 0 30px rgba(255,215,0,0.4)}
.amount-tag{position:absolute;top:-8px;right:-8px;background:#ffd700;color:#000;font-size:12px;padding:4px 8px;border-radius:12px;font-weight:bold;box-shadow:0 2px 8px rgba(0,0,0,0.3)}

.red{background:linear-gradient(135deg,#e74c3c,#c0392b)}
.green{background:linear-gradient(135deg,#27ae60,#229954)}
.yellow{background:linear-gradient(135deg,#f39c12,#e67e22);color:#000;text-shadow:0 1px 3px rgba(0,0,0,0.3)}
.blue{background:linear-gradient(135deg,#3498db,#2980b9)}
.purple{background:linear-gradient(135deg,#9b59b6,#8e44ad)}
.orange{background:linear-gradient(135deg,#e67e22,#d35400)}
.pink{background:linear-gradient(135deg,#e91e63,#c2185b)}
.cyan{background:linear-gradient(135deg,#1abc9c,#16a085)}

.amount-box{display:flex;justify-content:center;gap:15px;margin:20px;flex-wrap:wrap}
.amount-btn{padding:15px 20px;border:none;border-radius:12px;font-size:16px;font-weight:bold;cursor:pointer;background:rgba(255,255,255,0.1);color:#fff;transition:all 0.3s ease;backdrop-filter:blur(5px)}
.amount-btn:hover{background:rgba(255,255,255,0.2);transform:translateY(-2px)}
.amount-btn.active{background:#ffd700;color:#000;box-shadow:0 0 20px rgba(255,215,0,0.4)}
#customAmount{border:2px solid rgba(255,255,255,0.3)}
#customAmount:focus{outline:none;border-color:#ffd700}
#customAmount::placeholder{color:rgba(255,255,255,0.6)}

.betbtn{margin:20px auto;display:block;width:80%;max-width:300px;padding:18px;font-size:20px;font-weight:bold;background:linear-gradient(135deg,#ffd700,#ffb700);color:#000;border:none;border-radius:15px;cursor:pointer;transition:all 0.3s ease;box-shadow:0 4px 15px rgba(255,215,0,0.3)}
.betbtn:hover{transform:translateY(-2px);box-shadow:0 8px 25px rgba(255,215,0,0.4)}
.betbtn:disabled{opacity:0.5;cursor:not-allowed}

.modal{position:fixed;inset:0;background:rgba(0,0,0,0.8);display:none;align-items:center;justify-content:center;backdrop-filter:blur(5px);z-index:1000;padding:10px}
.modal-box{background:linear-gradient(135deg,#2c3e50,#34495e);padding:20px;border-radius:20px;text-align:center;width:90%;max-width:400px;max-height:80vh;overflow-y:auto;box-shadow:0 20px 60px rgba(0,0,0,0.5);position:relative}
.modal-box h2{color:#ffd700;margin-bottom:15px;font-size:20px}
.modal-box button{margin:8px;padding:12px 20px;border:none;border-radius:10px;font-weight:bold;cursor:pointer;transition:all 0.3s ease;font-size:14px}
.btn-primary{background:#ffd700;color:#000}.btn-secondary{background:#6c757d;color:#fff}
.btn-primary:hover{background:#ffed4e}.btn-secondary:hover{background:#5a6268}

.close-btn{position:absolute;top:8px;right:12px;background:none;border:none;color:#fff;font-size:20px;cursor:pointer;padding:5px;border-radius:50%;width:30px;height:30px;display:flex;align-items:center;justify-content:center;transition:all 0.3s ease}
.close-btn:hover{background:rgba(255,255,255,0.1);color:#ffd700}

/* Result Modal Specific */
#resultModal .modal-box{max-width:450px}
#resText{font-size:14px;line-height:1.6;margin:15px 0;text-align:left}
#resText strong{color:#ffd700}
#winnerDisplay{margin:15px 0;padding:12px;border-radius:10px;font-weight:bold;font-size:16px !important;text-align:center}

/* Custom Alert Dialog */
.alert-modal{position:fixed;inset:0;background:rgba(0,0,0,0.9);display:none;align-items:center;justify-content:center;backdrop-filter:blur(8px);z-index:1001;padding:10px}
.alert-box{background:linear-gradient(135deg,#34495e,#2c3e50);padding:20px;border-radius:15px;text-align:center;width:90%;max-width:350px;box-shadow:0 15px 50px rgba(0,0,0,0.7);border:2px solid rgba(255,215,0,0.3)}
.alert-box.success{border-color:rgba(39,174,96,0.5)}.alert-box.error{border-color:rgba(231,76,60,0.5)}
.alert-icon{font-size:40px;margin-bottom:12px}
.alert-icon.success{color:#27ae60}.alert-icon.error{color:#e74c3c}
.alert-title{color:#ffd700;font-size:16px;font-weight:bold;margin-bottom:8px}
.alert-message{color:#ecf0f1;font-size:13px;line-height:1.5;margin-bottom:15px}
.alert-btn{background:#ffd700;color:#000;border:none;padding:10px 20px;border-radius:8px;font-weight:bold;cursor:pointer;transition:all 0.3s ease;font-size:14px}
.alert-btn:hover{background:#ffed4e;transform:translateY(-1px)}

/* Mobile Responsiveness - FIXED FOR PROPER MOBILE VIEW */
@media (max-width: 768px) {
 body{font-size:16px}
 
 .header{padding:15px 20px;flex-direction:row;gap:10px}
 .header-left,.header-center,.header-right{flex:1}
 .header-center{text-align:center}
 .header-right{text-align:right}
 
 .colors{grid-template-columns:repeat(2,1fr);gap:20px;padding:25px;max-width:100%}
 .color{padding:30px 20px;font-size:18px;min-height:90px;border-radius:15px}
 
 .amount-box{gap:15px;margin:25px 20px;justify-content:center;flex-wrap:wrap}
 .amount-btn{padding:18px 25px;font-size:18px;min-width:100px;border-radius:12px}
 #customAmount{padding:18px 25px;font-size:18px;width:180px}
 
 .timer{font-size:48px;font-weight:bold}
 .timer-section{padding:25px;margin:15px;border-radius:15px}
 
 .betbtn{width:80%;max-width:350px;font-size:22px;padding:20px;margin:25px auto;border-radius:15px}
 
 /* Modal Responsive */
 .modal{padding:15px}
 .modal-box{width:85%;padding:25px;max-height:75vh;border-radius:20px}
 .modal-box h2{font-size:22px;margin-bottom:20px}
 .modal-box button{padding:15px 25px;font-size:16px;margin:10px;border-radius:10px}
 
 #resText{font-size:16px;line-height:1.7}
 #winnerDisplay{font-size:18px !important;padding:15px;border-radius:12px}
 
 .alert-box{width:85%;padding:25px;border-radius:18px}
 .alert-icon{font-size:45px;margin-bottom:15px}
 .alert-title{font-size:19px;margin-bottom:12px}
 .alert-message{font-size:15px;line-height:1.5}
 .alert-btn{padding:12px 25px;font-size:16px;border-radius:10px}
 
 .top-bar{top:15px;right:15px}
 .back-btn{padding:10px 18px;font-size:16px;border-radius:10px}
}

@media (max-width: 480px) {
 body{font-size:14px}
 
 .balance{font-size:20px;font-weight:bold}
 .user-info{font-size:16px}
 
 .colors{grid-template-columns:1fr 1fr;gap:15px;padding:20px}
 .color{padding:25px 15px;font-size:16px;min-height:80px;border-radius:12px}
 
 .amount-box{gap:12px;margin:20px 15px;flex-wrap:wrap}
 .amount-btn{padding:15px 20px;font-size:16px;min-width:85px;flex:1;max-width:120px}
 #customAmount{padding:15px 20px;font-size:16px;width:160px}
 
 .timer{font-size:40px}
 .timer-section{padding:20px;margin:12px;border-radius:12px}
 .betbtn{font-size:20px;padding:18px;width:85%;border-radius:12px}
 
 /* Modal for Small Screens */
 .modal{padding:10px}
 .modal-box{width:90%;padding:20px;border-radius:15px}
 .modal-box h2{font-size:20px;margin-bottom:15px}
 .modal-box button{padding:12px 20px;font-size:15px;margin:8px}
 
 #resText{font-size:15px;line-height:1.6}
 #winnerDisplay{font-size:16px !important;padding:12px}
 
 .close-btn{width:35px;height:35px;font-size:20px;top:10px;right:12px}
 
 .alert-box{width:90%;padding:20px}
 .alert-icon{font-size:40px}
 .alert-title{font-size:18px}
 .alert-message{font-size:14px}
 .alert-btn{padding:12px 22px;font-size:15px}
}

/* Landscape Mobile */
@media (max-width: 768px) and (orientation: landscape) {
 .modal-box{max-height:90vh;overflow-y:auto}
 .timer-section{padding:10px}
 .timer{font-size:24px}
 .colors{grid-template-columns:repeat(4,1fr);gap:8px}
 .color{padding:12px;font-size:12px}
}

/* Touch-friendly improvements */
@media (max-width: 768px) {
 .color{
  min-height:60px;
  display:flex;
  align-items:center;
  justify-content:center;
  touch-action:manipulation;
 }
 
 .amount-btn{
  min-height:44px;
  touch-action:manipulation;
 }
 
 .betbtn{
  min-height:50px;
  touch-action:manipulation;
 }
 
 /* Prevent zoom on input focus */
 input, select, textarea {
  font-size:16px;
 }
}

.logout-btn{position:absolute;top:20px;right:20px;background:rgba(220,53,69,0.8);color:#fff;border:none;padding:8px 15px;border-radius:8px;cursor:pointer;text-decoration:none;font-size:14px}
.logout-btn:hover{background:rgba(220,53,69,1)}


.top-bar{
 position:fixed;
 top:10px;
 right:10px;
 z-index:999;
}

.back-btn{
 background:#e5e7eb;
 border:none;
 padding:8px 14px;
 border-radius:10px;
 font-size:14px;
 font-weight:600;
 cursor:pointer;
 box-shadow:0 4px 10px rgba(0,0,0,.15);
}

.back-btn:hover{
 background:#d1d5db;
}




</style>
</head>
<body>


<!-- TOP RIGHT BACK BUTTON -->
<div class="top-bar">
  <button class="back-btn" onclick="goHome()">⬅ Back to Home</button>
</div>







<div class="header">
<div class="header-left">
    <div class="user-info">👤 Player ID: <?= $user_id ?></div>
</div>
<div class="header-center">
    <div class="balance">💰 ₹ <span id="wallet"><?= $user['wallet'] ?></span></div>
</div>

<div class="header-right">
    <!-- <a href="logout.php" class="logout-btn">Logout</a> -->
</div>


</div>

<div class="timer-section">
    <div>⏰ Time Remaining</div>
    <div class="timer" id="time">30</div>
    <div id="gameStatus">Place your bets now!</div>
    <div style="margin-top:10px;padding:8px;background:rgba(255,215,0,0.2);border-radius:8px;border:1px solid #ffd700;">
        <div style="font-size:12px;color:#ffd700;font-weight:bold;">🎯 WIN = 7X YOUR BET!</div>
        <div style="font-size:11px;opacity:0.8;">Bet ₹10 → Win ₹70 | Bet ₹100 → Win ₹700</div>
    </div>
</div>

<div class="colors">
<?php foreach($colors as $c): ?>
<div class="color <?= $c ?>" onclick="pick(this,'<?= $c ?>')"><?= strtoupper($c) ?></div>
<?php endforeach; ?>
</div>

<div class="amount-box">
<button class="amount-btn" onclick="amt(this,10)">₹10</button>
<button class="amount-btn" onclick="amt(this,50)">₹50</button>
<button class="amount-btn" onclick="amt(this,100)">₹100</button>
<button class="amount-btn" onclick="amt(this,500)">₹500</button>
<input type="number" id="customAmount" class="amount-btn" placeholder="Custom ₹" min="10" step="10" style="width:150px;text-align:center;" onchange="setCustomAmount(this.value)">
</div>

<button class="betbtn" onclick="confirmBet()">BET NOW</button>

<!-- BET MODAL -->
<div class="modal" id="betModal">
<div class="modal-box">
<button class="close-btn" onclick="closeBet()">&times;</button>
<h2>🎯 Confirm Your Bet</h2>
<div id="betText" style="margin:20px 0;font-size:16px;line-height:1.6"></div>
<button class="btn-primary" onclick="placeBet()">✅ Place Bet</button>
<button class="btn-secondary" onclick="closeBet()">❌ Cancel</button>
</div>
</div>

<!-- RESULT MODAL -->
<div class="modal" id="resultModal">
<div class="modal-box">
<button class="close-btn" onclick="closeResult()">&times;</button>
<h2 id="resTitle"></h2>
<div id="resText" style="margin:20px 0;font-size:16px;line-height:1.8"></div>
<div id="winnerDisplay" style="margin:15px 0;padding:15px;border-radius:10px;font-weight:bold;font-size:18px"></div>
<button class="btn-primary" onclick="closeResult()">🎮 Play Again</button>
</div>
</div>

<!-- CUSTOM ALERT MODAL -->
<div class="alert-modal" id="alertModal">
<div class="alert-box" id="alertBox">
<div class="alert-icon" id="alertIcon">✅</div>
<div class="alert-title" id="alertTitle">Success</div>
<div class="alert-message" id="alertMessage">Your bet has been placed successfully!</div>
<button class="alert-btn" onclick="closeAlert()">OK</button>
</div>
</div>

<script>
let selectedColors=[],colorAmounts={},lastColorEl=null;
let historyColors=[],time=30;
let timerInterval=null,resultRunning=false;
let game_id=<?= $game_id ?>;
let bettingEnabled=true;
let currentAmount=0;

/* SIMPLE TIMER - BACK TO WORKING VERSION */
startTimer();
function startTimer(){
 // Reset timer to 30 seconds
 time = 30;
 
 bettingEnabled=true;
 document.getElementById("gameStatus").innerText="Place your bets now!";
 document.querySelector(".betbtn").disabled=false;
 
 timerInterval=setInterval(()=>{
  document.getElementById("time").innerText=time;
  
  // Warning animation for last 10 seconds
  if(time<=10){
   document.getElementById("time").classList.add("warning");
   document.getElementById("gameStatus").innerText="⚠️ Hurry up! Betting closes soon!";
  }
  
  // Stop betting 3 seconds before result
  if(time<=3){
   bettingEnabled=false;
   document.getElementById("gameStatus").innerText="🚫 Betting Closed - Calculating Result...";
   document.querySelector(".betbtn").disabled=true;
  }
  
  time--;
  if(time<0){
   clearInterval(timerInterval);
   // Call result function like before
   result();
  }
 },1000);
}

function pick(el,color){
 if(!bettingEnabled){
  showAlert("error", "Betting Closed", "⚠️ Betting is closed for this round!");
  return;
 }
 
 if(!currentAmount){
  showAlert("error", "Select Amount", "💰 Please select bet amount first!");
  return;
 }
 
 if(selectedColors.includes(color)){
  // Remove color
  selectedColors=selectedColors.filter(c=>c!==color);
  delete colorAmounts[color];
  el.classList.remove("active");
  el.querySelector(".amount-tag")?.remove();
 }else{
  // Add color with current amount
  selectedColors.push(color);
  colorAmounts[color] = currentAmount;
  el.classList.add("active");
  
  // Add amount tag
  el.querySelector(".amount-tag")?.remove();
  let tag=document.createElement("div");
  tag.className="amount-tag";
  tag.innerText="₹"+currentAmount;
  el.appendChild(tag);
 }
 lastColorEl=el;
}

function amt(btn,val){
 currentAmount=val;
 document.querySelectorAll(".amount-btn").forEach(b=>b.classList.remove("active"));
 if(btn.tagName === 'BUTTON') {
  btn.classList.add("active");
 }
 // Clear custom input if button clicked
 document.getElementById("customAmount").value = "";
}

function setCustomAmount(val){
 val = parseInt(val);
 if(val && val >= 10){
  currentAmount = val;
  // Remove active class from buttons
  document.querySelectorAll(".amount-btn").forEach(b=>b.classList.remove("active"));
  // Highlight custom input
  document.getElementById("customAmount").classList.add("active");
 } else {
  showAlert("error", "Invalid Amount", "⚠️ Minimum bet amount is ₹10!");
  document.getElementById("customAmount").value = "";
 }
}

function confirmBet(){
 if(!bettingEnabled){
  showAlert("error", "Betting Closed", "⚠️ Betting is closed for this round!");
  return;
 }
 
 if(!selectedColors.length){
  showAlert("error", "No Colors Selected", "🎨 Please select at least one color!");
  return;
 }
 
 let totalAmount = Object.values(colorAmounts).reduce((sum, amt) => sum + amt, 0);
 let currentWallet = parseInt(document.getElementById("wallet").innerText);
 
 if(totalAmount > currentWallet){
  showAlert("error", "Insufficient Balance", `💸 You need ₹${totalAmount} but have only ₹${currentWallet}`);
  return;
 }
 
 // Create bet display
 let betDisplay = "<strong>Selected Bets:</strong><br>";
 selectedColors.forEach(color => {
  betDisplay += `🎨 ${color.toUpperCase()} : ₹${colorAmounts[color]}<br>`;
 });
 betDisplay += `<br><strong>Total Amount: ₹${totalAmount}</strong>`;
 
 document.getElementById("betText").innerHTML = betDisplay;
 document.getElementById("betModal").style.display="flex";
}

function closeBet(){document.getElementById("betModal").style.display="none";}

function placeBet(){
 console.log("🎯 PLACING BET:");
 console.log("Selected Colors:", selectedColors);
 console.log("Color Amounts:", colorAmounts);
 
 // Prepare bet data
 let betData = "action=place_bet&game_id=" + game_id;
 selectedColors.forEach(color => {
  betData += "&bets[]=" + encodeURIComponent(JSON.stringify({
   color: color,
   amount: colorAmounts[color]
  }));
 });
 
 console.log("Bet Data:", betData);
 
 fetch("game.php",{

  method:"POST",
  headers:{'Content-Type':'application/x-www-form-urlencoded'},
  body: betData
 })
 .then(r=>r.json())
 .then(d=>{
  console.log("🎯 BET RESPONSE:");
  console.log("Status:", d.status);
  console.log("Message:", d.msg);
  console.log("Bets Placed:", d.bets_placed);
  console.log("Total Amount:", d.total_amount);
  console.log("New Wallet:", d.wallet);
  console.log("Inserted Bets:", d.inserted_bets);
  
  if(d.status==="success"){
   document.getElementById("wallet").innerText=d.wallet;
   closeBet();
   
   // Clear selected colors and amounts after successful bet
   clearSelections();
   
   // Disable further betting until result
   bettingEnabled = false;
   document.querySelector(".betbtn").disabled = true;
   document.getElementById("gameStatus").innerText = "✅ Bet Placed! Wait for result...";
   
   showAlert("success", "Bet Placed!", `✅ ${d.bets_placed} bets placed for ₹${d.total_amount}!`);
  }else{
   console.log("❌ BET FAILED:", d.msg);
   showAlert("error", "Bet Failed", "❌ " + d.msg);
  }
 })
 .catch(e=>{
  console.error("❌ BET ERROR:", e);
  showAlert("error", "Network Error", "❌ Network error! Please try again.");
 });
}

// Function to clear all selections
function clearSelections(){
 selectedColors = [];
 colorAmounts = {};
 currentAmount = 0;
 
 // Clear UI
 document.querySelectorAll(".color").forEach(c=>{
  c.classList.remove("active");
  c.querySelector(".amount-tag")?.remove();
 });
 document.querySelectorAll(".amount-btn").forEach(b=>b.classList.remove("active"));
}

// Custom Alert Function
function showAlert(type, title, message) {
 const modal = document.getElementById("alertModal");
 const box = document.getElementById("alertBox");
 const icon = document.getElementById("alertIcon");
 const titleEl = document.getElementById("alertTitle");
 const messageEl = document.getElementById("alertMessage");
 
 // Set content
 titleEl.textContent = title;
 messageEl.textContent = message;
 
 // Set type styling
 if(type === "success") {
  box.className = "alert-box success";
  icon.textContent = "✅";
  icon.className = "alert-icon success";
 } else {
  box.className = "alert-box error";
  icon.textContent = "❌";
  icon.className = "alert-icon error";
 }
 
 modal.style.display = "flex";
}

function closeAlert() {
 document.getElementById("alertModal").style.display = "none";
}

function result(){
 if(resultRunning) return;
 resultRunning=true;
 
 document.getElementById("gameStatus").innerText="🎲 Generating Result...";
 
 console.log("🎯 RESULT REQUEST - Game ID:", game_id);
 
 fetch("game.php",{
  method:"POST",
  headers:{'Content-Type':'application/x-www-form-urlencoded'},
  body:"action=result&game_id="+game_id
 })
 .then(r=>r.json())
 .then(d=>{
  console.log("✅ Result received:", d);
  
  // Update wallet
  document.getElementById("wallet").innerText = d.wallet;
  game_id = d.new_game_id;
  
  // Result display
  let winAmount = parseInt(d.win_amount) || 0;
  let isWin = winAmount > 0;
  
  document.getElementById("resTitle").innerHTML = isWin ? 
   "🎉 CONGRATULATIONS!" : "😔 Better Luck Next Time!";
  
  let resultText = `<div style="text-align:left;">`;
  resultText += `<p><strong>🏆 Winning Color:</strong> ${d.winner.toUpperCase()}</p>`;
  resultText += `<p><strong>💰 Total Winnings:</strong> ₹${winAmount}</p>`;
  resultText += `<p><strong>🎯 Total Bet:</strong> ₹${d.total_bet}</p>`;
  resultText += `<p><strong>💳 Updated Balance:</strong> ₹${d.wallet}</p>`;
  resultText += `</div>`;
  
  document.getElementById("resText").innerHTML = resultText;
  
  // Winner display with color styling
  let winnerDisplay = document.getElementById("winnerDisplay");
  winnerDisplay.className = `${d.winner}`;
  winnerDisplay.innerHTML = `�T WINNER: ${d.winner.toUpperCase()}`;
  winnerDisplay.style.fontSize = window.innerWidth < 480 ? "14px" : "18px";
  winnerDisplay.style.textAlign = "center";
  winnerDisplay.style.padding = window.innerWidth < 480 ? "8px" : "12px";
  winnerDisplay.style.borderRadius = "10px";
  winnerDisplay.style.fontWeight = "bold";
  
  document.getElementById("resultModal").style.display="flex";
  resultRunning=false;
 })
 .catch(e=>{
  console.error("❌ Result Error:", e);
  resultRunning=false;
  window.location.reload();
 });
}

function closeResult(){
 document.getElementById("resultModal").style.display="none";
 resultRunning=false;
 
 // Simple page refresh like before
 console.log("🔄 Refreshing page...");
 window.location.reload();
}

// Prevent page refresh during game
window.addEventListener('beforeunload', function(e) {
 if(bettingEnabled && selectedColors.length > 0) {
  e.preventDefault();
  e.returnValue = 'You have active bets. Are you sure you want to leave?';
 }
});

// Prevent browser back button after logout
window.addEventListener('pageshow', function(event) {
 if(event.persisted) {
  // Page loaded from cache, check if user is still logged in
  fetch('dashboard.php', {method: 'HEAD'})
  .then(response => {
   if(response.redirected || response.status === 302) {
    // User not logged in, redirect to login
    window.location.replace('login.php');
   }
  })
  .catch(() => {
   // Network error, redirect to login to be safe
   window.location.replace('login.php');
  });
 }
});

// Disable browser back button functionality
history.pushState(null, null, location.href);
window.addEventListener('popstate', function() {
 history.pushState(null, null, location.href);
});

// Close modal on outside click
document.addEventListener('click', function(e) {
 if(e.target.classList.contains('modal')) {
  if(e.target.id === 'betModal') closeBet();
  if(e.target.id === 'resultModal') closeResult();
 }
 if(e.target.classList.contains('alert-modal')) {
  closeAlert();
 }
});

// Escape key to close modals
document.addEventListener('keydown', function(e) {
 if(e.key === 'Escape') {
  if(document.getElementById('betModal').style.display === 'flex') closeBet();
  if(document.getElementById('resultModal').style.display === 'flex') closeResult();
  if(document.getElementById('alertModal').style.display === 'flex') closeAlert();
 }
});



function goHome(){
 window.location.href = "dashboard.php";
}




</script>

</body>
</html>
