<?php
$page_title = "Result History";
include("includes/header.php");

// Get selected date (default to today)
$selected_date = isset($_GET['date']) && !empty($_GET['date']) ? $_GET['date'] : date('Y-m-d');

// Validate date format
if(!preg_match('/^\d{4}-\d{2}-\d{2}$/', $selected_date)) {
    $selected_date = date('Y-m-d');
}

// Date range for query
$date_start = $selected_date . ' 00:00:00';
$date_end = $selected_date . ' 23:59:59';

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 25;
$offset = ($page - 1) * $limit;

// Get total count for selected date
$total_query = mysqli_query($conn, "
    SELECT COUNT(*) as total 
    FROM games 
    WHERE status='completed' 
    AND winning_color IS NOT NULL 
    AND completed_at BETWEEN '$date_start' AND '$date_end'
");
$total_results = mysqli_fetch_assoc($total_query)['total'];
$total_pages = ceil($total_results / $limit);

// Get game results for selected date
$results_query = mysqli_query($conn, "
    SELECT id, winning_color, completed_at,
           (SELECT COUNT(*) FROM bets WHERE game_id = games.id) as total_bets,
           (SELECT COUNT(*) FROM bets WHERE game_id = games.id AND status = 'win') as winners,
           (SELECT SUM(amount) FROM bets WHERE game_id = games.id) as total_wagered,
           (SELECT SUM(win_amount) FROM bets WHERE game_id = games.id) as total_winnings
    FROM games 
    WHERE status='completed' 
    AND winning_color IS NOT NULL
    AND completed_at BETWEEN '$date_start' AND '$date_end'
    ORDER BY completed_at DESC
    LIMIT $limit OFFSET $offset
");
?>

<div class="container">
    <!-- Stats Overview -->
    <div class="content-box">
        <h2 class="section-title">📊 Color Statistics for <?= date('M j, Y', strtotime($selected_date)) ?></h2>
        
        <?php
        $stats = mysqli_fetch_assoc(mysqli_query($conn, "
            SELECT 
                COUNT(*) as total_games,
                SUM(CASE WHEN winning_color='red' THEN 1 ELSE 0 END) as red_wins,
                SUM(CASE WHEN winning_color='green' THEN 1 ELSE 0 END) as green_wins,
                SUM(CASE WHEN winning_color='yellow' THEN 1 ELSE 0 END) as yellow_wins,
                SUM(CASE WHEN winning_color='blue' THEN 1 ELSE 0 END) as blue_wins,
                SUM(CASE WHEN winning_color='purple' THEN 1 ELSE 0 END) as purple_wins,
                SUM(CASE WHEN winning_color='orange' THEN 1 ELSE 0 END) as orange_wins,
                SUM(CASE WHEN winning_color='pink' THEN 1 ELSE 0 END) as pink_wins,
                SUM(CASE WHEN winning_color='cyan' THEN 1 ELSE 0 END) as cyan_wins
            FROM games 
            WHERE status='completed' 
            AND winning_color IS NOT NULL
            AND completed_at BETWEEN '$date_start' AND '$date_end'
        "));
        ?>
        
        <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(150px,1fr));gap:15px;">
            <?php
            $colors = ['red', 'green', 'yellow', 'blue', 'purple', 'orange', 'pink', 'cyan'];
            foreach($colors as $color):
                $wins = $stats[$color . '_wins'];
                $percentage = $stats['total_games'] > 0 ? round(($wins / $stats['total_games']) * 100, 1) : 0;
            ?>
                <div class="card" style="text-align:center;">
                    <div style="width:40px;height:40px;border-radius:50%;background:<?= getColorCode($color) ?>;margin:0 auto 10px;"></div>
                    <h4 style="color:<?= getColorCode($color) ?>;margin-bottom:5px;"><?= strtoupper($color) ?></h4>
                    <p style="font-size:18px;font-weight:700;margin-bottom:5px;"><?= $wins ?></p>
                    <p style="font-size:12px;opacity:0.7;"><?= $percentage ?>%</p>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Results History -->
    <div class="content-box">
        <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;flex-wrap:wrap;gap:15px;">
            <h2 class="section-title">🎯 Game Results for <?= date('M j, Y', strtotime($selected_date)) ?></h2>
            
            <!-- Date Filter -->
            <form method="get" style="display:flex;align-items:center;gap:10px;">
                <input type="date" name="date" value="<?= $selected_date ?>" class="form-input" style="width:auto;min-width:150px;">
                <button type="submit" class="btn">📊 View Results</button>
                <a href="?date=<?= date('Y-m-d') ?>" class="btn btn-secondary">📅 Today</a>
                <?php if(isset($_GET['page']) && $_GET['page'] > 1): ?>
                    <input type="hidden" name="page" value="1">
                <?php endif; ?>
            </form>
        </div>
        
        <?php if($results_query && mysqli_num_rows($results_query) > 0): ?>
            <table class="table result-history-table">
                <thead>
                    <tr>
                        <th>Time</th>
                        <th>Winning Color</th>
                        <th>Total Bets</th>
                        <th>Winners</th>
                        <th>Total Wagered</th>
                        <th>Total Winnings</th>
                        <th>House Edge</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while($result = mysqli_fetch_assoc($results_query)): 
                        // Handle null values
                        $total_bets = $result['total_bets'] ?? 0;
                        $winners = $result['winners'] ?? 0;
                        $total_wagered = $result['total_wagered'] ?? 0;
                        $total_winnings = $result['total_winnings'] ?? 0;
                        
                        $house_edge = $total_wagered > 0 ? 
                            round((($total_wagered - $total_winnings) / $total_wagered) * 100, 1) : 0;
                    ?>
                        <tr>
                            <td style="font-weight:700;"><?= date('H:i:s', strtotime($result['completed_at'])) ?></td>
                            <td>
                                <div style="display:flex;align-items:center;gap:10px;">
                                    <span style="width:25px;height:25px;border-radius:50%;background:<?= getColorCode($result['winning_color']) ?>;display:inline-block;"></span>
                                    <strong><?= strtoupper($result['winning_color']) ?></strong>
                                </div>
                            </td>
                            <td style="font-weight:700;"><?= number_format($total_bets) ?></td>
                            <td style="color:#27ae60;font-weight:700;"><?= number_format($winners) ?></td>
                            <td style="font-weight:700;">₹<?= number_format($total_wagered, 2) ?></td>
                            <td style="color:#f39c12;font-weight:700;">₹<?= number_format($total_winnings, 2) ?></td>
                            <td style="color:<?= $house_edge > 0 ? '#27ae60' : '#e74c3c' ?>;font-weight:700;">
                                <?= $house_edge ?>%
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
            
            <!-- Pagination -->
            <?php if($total_pages > 1): ?>
                <div style="text-align:center;margin-top:30px;">
                    <?php for($i = 1; $i <= $total_pages; $i++): ?>
                        <a href="?page=<?= $i ?>&date=<?= $selected_date ?>" class="btn <?= $i == $page ? '' : 'btn-secondary' ?>" style="margin:0 5px;">
                            <?= $i ?>
                        </a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
            
        <?php else: ?>
            <div class="empty-state">
                <div class="empty-state-icon">🎯</div>
                <div class="empty-state-title">No Results Available</div>
                <div class="empty-state-text">
                    <?php if($selected_date == date('Y-m-d')): ?>
                        No game results for today yet. Results will appear here once games are completed.
                    <?php else: ?>
                        No game results found for <?= date('M j, Y', strtotime($selected_date)) ?>
                    <?php endif; ?>
                </div>
                <div style="margin-top:20px;">
                    <a href="game.php" class="btn" style="margin-right:10px;">🎯 Play Game</a>
                    <a href="?date=<?= date('Y-m-d') ?>" class="btn btn-secondary">📅 View Today</a>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Recent Winners -->
    <div class="content-box">
        <h2 class="section-title">🏆 Big Winners for <?= date('M j, Y', strtotime($selected_date)) ?></h2>
        
        <?php
        $big_winners = mysqli_query($conn, "
            SELECT u.name, b.amount, b.win_amount, b.color, g.winning_color, b.created_at
            FROM bets b
            JOIN users u ON b.user_id = u.id
            JOIN games g ON b.game_id = g.id
            WHERE b.status = 'win' 
            AND b.win_amount >= 500
            AND b.created_at BETWEEN '$date_start' AND '$date_end'
            ORDER BY b.win_amount DESC
            LIMIT 10
        ");
        
        if($big_winners && mysqli_num_rows($big_winners) > 0):
        ?>
            <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(300px,1fr));gap:20px;">
                <?php while($winner = mysqli_fetch_assoc($big_winners)): ?>
                    <div class="card" style="background:rgba(255,215,0,0.1);border-color:#ffd700;">
                        <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:10px;">
                            <h4 style="color:#ffd700;margin:0;">🏆 <?= htmlspecialchars(substr($winner['name'], 0, 1) . str_repeat('*', strlen($winner['name'])-2) . substr($winner['name'], -1)) ?></h4>
                            <span style="font-size:12px;opacity:0.7;"><?= date('H:i', strtotime($winner['created_at'])) ?></span>
                        </div>
                        <p style="margin:5px 0;">
                            Bet: <span style="color:<?= getColorCode($winner['color']) ?>;font-weight:700;"><?= strtoupper($winner['color']) ?></span>
                            | Won: <span style="color:<?= getColorCode($winner['winning_color']) ?>;font-weight:700;"><?= strtoupper($winner['winning_color']) ?></span>
                        </p>
                        <p style="margin:5px 0;">
                            Wagered: <strong>₹<?= number_format($winner['amount'], 2) ?></strong>
                        </p>
                        <p style="margin:0;color:#27ae60;font-weight:700;font-size:18px;">
                            Won: ₹<?= number_format($winner['win_amount'], 2) ?>
                        </p>
                    </div>
                <?php endwhile; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <div class="empty-state-icon">🏆</div>
                <div class="empty-state-title">No Big Winners</div>
                <div class="empty-state-text">
                    <?php if($selected_date == date('Y-m-d')): ?>
                        No big winners (₹500+) today yet
                    <?php else: ?>
                        No big winners found for <?= date('M j, Y', strtotime($selected_date)) ?>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php
function getColorCode($color) {
    $colors = [
        'red' => '#ef4444',
        'green' => '#22c55e', 
        'yellow' => '#facc15',
        'blue' => '#3b82f6',
        'purple' => '#a855f7',
        'orange' => '#fb923c',
        'pink' => '#ec4899',
        'cyan' => '#06b6d4'
    ];
    return $colors[strtolower($color)] ?? '#666';
}

include("includes/footer.php");
?>