<?php
// Game Cron Job - Call this every minute
// This can be set up as a real cron job or called via AJAX

include("config/db.php");
header('Content-Type: application/json');

/* ================= BUSINESS ALGORITHM FOR CRON ================= */
function calculateSmartWinnerCron($conn, $game_id) {
    // Get all pending bets for this game from ALL users
    $all_bets = mysqli_query($conn, "
        SELECT color, SUM(amount) as total_amount, COUNT(*) as bet_count
        FROM bets 
        WHERE game_id = $game_id AND status = 'pending'
        GROUP BY color
        ORDER BY total_amount DESC
    ");
    
    $color_stats = [];
    $total_wagered = 0;
    $colors = ['red','green','yellow','blue','purple','orange','pink','cyan'];
    
    // Initialize all colors with 0
    foreach($colors as $color) {
        $color_stats[$color] = ['amount' => 0, 'count' => 0];
    }
    
    // Fill actual bet data
    if($all_bets && mysqli_num_rows($all_bets) > 0) {
        while($bet = mysqli_fetch_assoc($all_bets)) {
            $color_stats[$bet['color']] = [
                'amount' => (float)$bet['total_amount'],
                'count' => (int)$bet['bet_count']
            ];
            $total_wagered += (float)$bet['total_amount'];
        }
    }
    
    // If no bets, return random color
    if($total_wagered == 0) {
        return $colors[array_rand($colors)];
    }
    
    // BUSINESS ALGORITHM: Calculate potential losses for each color
    $potential_losses = [];
    $payout_multiplier = 7; // 7x payout
    
    foreach($colors as $color) {
        $bet_amount = $color_stats[$color]['amount'];
        $potential_payout = $bet_amount * $payout_multiplier;
        $house_loss = $potential_payout - $total_wagered;
        
        $potential_losses[$color] = [
            'bet_amount' => $bet_amount,
            'potential_payout' => $potential_payout,
            'house_loss' => $house_loss,
            'profit_margin' => (($total_wagered - $potential_payout) / $total_wagered) * 100
        ];
    }
    
    // Sort colors by house profit (ascending house loss = more profit)
    uasort($potential_losses, function($a, $b) {
        return $a['house_loss'] <=> $b['house_loss'];
    });
    
    // SMART SELECTION ALGORITHM
    $profitable_colors = [];
    $neutral_colors = [];
    $loss_colors = [];
    
    foreach($potential_losses as $color => $stats) {
        if($stats['house_loss'] <= 0) {
            $profitable_colors[] = $color;
        } elseif($stats['house_loss'] <= ($total_wagered * 0.1)) {
            $neutral_colors[] = $color;
        } else {
            $loss_colors[] = $color;
        }
    }
    
    // SELECTION PROBABILITY (House always has advantage)
    $rand = mt_rand(1, 100);
    
    if($rand <= 70 && !empty($profitable_colors)) {
        return $profitable_colors[0];
    } elseif($rand <= 85 && !empty($profitable_colors)) {
        return $profitable_colors[array_rand($profitable_colors)];
    } elseif($rand <= 95 && !empty($neutral_colors)) {
        return $neutral_colors[array_rand($neutral_colors)];
    } else {
        if(!empty($loss_colors)) {
            return $loss_colors[array_rand($loss_colors)];
        } else {
            return $colors[array_rand($colors)];
        }
    }
}

$response = ['status' => 'success', 'message' => '', 'actions' => []];

try {
    // Handle get_user_result action
    if(isset($_POST['action']) && $_POST['action'] == 'get_user_result') {
        $game_id = (int)($_POST['game_id'] ?? 0);
        $winner = $_POST['winner'] ?? '';
        $user_id = (int)($_SESSION['user_id'] ?? 0);
        
        if($user_id && $game_id && $winner) {
            // Get user's bets for this game
            $user_bets = mysqli_query($conn, "
                SELECT * FROM bets 
                WHERE game_id=$game_id AND user_id=$user_id
            ");
            
            $total_bet = 0;
            $win_amount = 0;
            $winning_bets = [];
            
            while($bet = mysqli_fetch_assoc($user_bets)) {
                $total_bet += $bet['amount'];
                
                if(strtolower($bet['color']) === strtolower($winner)) {
                    $win_amount += $bet['win_amount'];
                    $winning_bets[] = [
                        'color' => $bet['color'],
                        'bet_amount' => $bet['amount'],
                        'win_amount' => $bet['win_amount']
                    ];
                }
            }
            
            // Get updated wallet
            $user = mysqli_fetch_assoc(mysqli_query($conn, "SELECT wallet FROM users WHERE id=$user_id"));
            
            $response = [
                'status' => 'success',
                'win_amount' => $win_amount,
                'total_bet' => $total_bet,
                'wallet' => (int)$user['wallet'],
                'winning_bets' => $winning_bets
            ];
        } else {
            $response = ['status' => 'error', 'message' => 'Invalid parameters'];
        }
        
        echo json_encode($response);
        exit;
    }

    // Regular game processing
    // Get current running game
    $running_game = mysqli_query($conn, "
        SELECT id, created_at 
        FROM games 
        WHERE status='running' 
        ORDER BY id DESC 
        LIMIT 1
    ");

    if($running_game && mysqli_num_rows($running_game) > 0) {
        $game = mysqli_fetch_assoc($running_game);
        $game_id = $game['id'];
        $created_time = strtotime($game['created_at']);
        $current_time = time();
        $elapsed_seconds = $current_time - $created_time;
        
        $response['current_game'] = $game_id;
        $response['elapsed_seconds'] = $elapsed_seconds;
        
        // Check if game should be completed (after 30 seconds)
        if($elapsed_seconds >= 30) {
            // 🎯 SMART BUSINESS ALGORITHM - Calculate optimal winner
            $colors = ['red', 'green', 'yellow', 'blue', 'purple', 'orange', 'pink', 'cyan'];
            $winner = calculateSmartWinnerCron($conn, $game_id);
            
            // Process all pending bets
            $bets_result = mysqli_query($conn, "
                SELECT * FROM bets 
                WHERE game_id=$game_id AND status='pending'
            ");
            
            $total_bets = mysqli_num_rows($bets_result);
            $winners_count = 0;
            $total_payout = 0;
            
            if($total_bets > 0) {
                while($bet = mysqli_fetch_assoc($bets_result)) {
                    $user_id = $bet['user_id'];
                    $bet_color = strtolower(trim($bet['color']));
                    $bet_amount = $bet['amount'];
                    $winner_color = strtolower(trim($winner));
                    
                    if($bet_color === $winner_color) {
                        // WIN - 7x payout (Business Algorithm)
                        $win_amount = $bet_amount * 7;
                        $total_payout += $win_amount;
                        $winners_count++;
                        
                        // Add to wallet
                        mysqli_query($conn, "UPDATE users SET wallet = wallet + $win_amount WHERE id = $user_id");
                        
                        // Update bet to win
                        mysqli_query($conn, "UPDATE bets SET status='win', win_amount=$win_amount WHERE id={$bet['id']}");
                    } else {
                        // LOSS
                        mysqli_query($conn, "UPDATE bets SET status='loss', win_amount=0 WHERE id={$bet['id']}");
                    }
                }
            }
            
            // Mark game as completed
            $winner_escaped = mysqli_real_escape_string($conn, $winner);
            mysqli_query($conn, "
                UPDATE games 
                SET status='completed', winning_color='$winner_escaped', completed_at=NOW() 
                WHERE id=$game_id
            ");
            
            // Create new running game
            mysqli_query($conn, "INSERT INTO games(status,created_at) VALUES('running',NOW())");
            $new_game_id = mysqli_insert_id($conn);
            
            $response['actions'][] = 'game_completed';
            $response['completed_game'] = $game_id;
            $response['winner'] = $winner;
            $response['total_bets'] = $total_bets;
            $response['winners_count'] = $winners_count;
            $response['total_payout'] = $total_payout;
            $response['new_game'] = $new_game_id;
            $response['message'] = "Game $game_id completed. Winner: $winner. New game: $new_game_id";
            
        } else {
            $remaining = max(0, min(30, 30 - $elapsed_seconds));
            $response['remaining_seconds'] = $remaining;
            $response['message'] = "Game $game_id running. $remaining seconds remaining.";
        }
        
    } else {
        // No running game, create one
        mysqli_query($conn, "INSERT INTO games(status,created_at) VALUES('running',NOW())");
        $new_game_id = mysqli_insert_id($conn);
        
        $response['actions'][] = 'game_created';
        $response['new_game'] = $new_game_id;
        $response['message'] = "New game created: $new_game_id";
    }
    
} catch(Exception $e) {
    $response['status'] = 'error';
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>